#region References

using System;
using System.Data;
using System.Linq;
using System.Text;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Xml;
using gov.va.med.vbecs.Common.Log;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;

#endregion

namespace gov.va.med.vbecs.DAL.HL7.Parsers
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//<Developer>Brian Tomlin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>4/30/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary>This class is used to parse and insert a new clinical order message from CPRS into VBECS.</summary>

	#endregion

	/// <summary>
	/// Class CprsNewOrder
	/// </summary>
	public class CprsNewOrder
	{
		#region Variables

		private static Hashtable _associatedInstitutions;

		private const string EXCEPTION_ASSOCIATED_INSTITUTION_LOOKUP = "Unable to find valid Associated Institutions information.  Please check configuration in VBECS Admin.";
		private const string LAST_UPDATE_USER_PARAMETER_VALUE = "CPRS";
		private const string SPECIMEN_ALREADY_IN_BLOOD_BANK = "A";
		private const string SPECIMEN_NOT_REQUIRED = "N";
		private const string SPECIMEN_REQUIRED = "R";
		private const string PRE_OP_ORDER = "P";
		private const string ORDERED_TEST = "test";
		private const string ORDERED_COMPONENT = "comp";
		private const string PENDING_ORDER_STATUS_CODE = "P";

		private const int ABORH_ORDERABLE_TEST_ID = 1;
		private const int ABORH_CPRS_ORDERABLE_ITEM_ID = 7;
		private const int ABS_ORDERABLE_TEST_ID = 3;
		private const int ABS_CPRS_ORDERABLE_ITEM_ID = 8;
		private const int DAT_ORDERABLE_TEST_ID = 4;
		private const int DAT_CPRS_ORDERABLE_ITEM_ID = 9;
		private const int TAS_ORDERABLE_TEST_ID = 7;
		private const int TAS_CPRS_ORDERABLE_ITEM_ID = 1;
		private const int TRW_ORDERABLE_TEST_ID = 10;
		private const int TRW_CPRS_ORDERABLE_ITEM_ID = 10;
		private const int RBC_COMPONENT_CLASS_ID = 1;
		private const int RBC_CPRS_ORDERABLE_ITEM_ID = 2;
		private const int FFP_COMPONENT_CLASS_ID = 2;
		private const int FFP_CPRS_ORDERABLE_ITEM_ID = 3;
		private const int CRYO_COMPONENT_CLASS_ID = 3;
		private const int CRYO_CPRS_ORDERABLE_ITEM_ID = 5;
		private const int PLT_COMPONENT_CLASS_ID = 4;
		private const int PLT_CPRS_ORDERABLE_ITEM_ID = 4;
		private const int OTHER_COMPONENT_CLASS_ID = 5;
		private const int OTHER_CPRS_ORDERABLE_ITEM_ID = 6;
		private const int WB_COMPONENT_CLASS_ID = 6;
		private const int WB_CPRS_ORDERABLE_ITEM_ID = 11;

		private Guid _patientGuid;
		private Guid _patientOrderGuid;
		private Guid _patientTreatmentGuid;

		private HL7OmgMessage _message;
		private HL7Interface _intParms;

		private BOL.Division _division;

		private string _orderTable;

		private DataTable _dtPatient;
		private DataRow _drPatient;
		private DataTable _dtPatientTreatment;
		private DataRow _drPatientTreatment;
		private DataTable _dtPatientOrder;
		private DataRow _drPatientOrder;
		private DataTable _dtOrderedTest;
		private DataRow _drOrderedTest;
		private DataTable _dtOrderedComponent;
		private DataRow _drOrderedComponent;

		private bool _patientExists = false;
		private bool _patientTreatmentExists = false;
		private bool _patientOrderExists = false;

        // Events Logger
        private readonly ILogger _eventsLogger =
            LogManager.Instance().LoggerLocator.GetLogger("SystemEvents");

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5820"> 
		///		<ExpectedInput>Non-null HL7OmgMessage type</ExpectedInput>
		///		<ExpectedOutput>Non-null instance of CprsNewOrder type.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5821"> 
		///		<ExpectedInput>Null message input parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor to create instance of CprsNewOrder and initialize required data tables.
		/// </summary>
		/// <param name="message"><see cref="HL7OmgMessage"/> representing a valid OMG^O19 HL7 message.</param>
		///  <param name="intParms"><see cref="HL7OmgMessage"/></param>
		public CprsNewOrder( HL7OmgMessage message, HL7Interface intParms )
		{
			if( message == null )
				throw( new ArgumentNullException( "message" ) );
			if( intParms == null )
				throw( new ArgumentNullException( "intParms" ) );

			this._message = message;
			this._intParms = intParms;

			InitializeDataTables();
		}
		
		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/18/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8496"> 
		///		<ExpectedInput>Valid AssociatedInstitution Data is available</ExpectedInput>
		///		<ExpectedOutput>CprsNewOrder object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8497"> 
		///		<ExpectedInput>AssociatedInstitution Data is not available</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor to create and initialize AssociatedInstitutions Hashtable; 
		/// to be used for caching AssociatedInstitutionCode -> DivisionCode mapping 
		/// for speedy lookup
		/// </summary>
		static CprsNewOrder()
		{
			_associatedInstitutions = new Hashtable();
			//
			DataTable dtAssociatedInstitutions = Common.StoredProcedure.GetData(STOREDPROC.GetAssociatedInstitutions.StoredProcName).Tables[0];
			//
			if ( dtAssociatedInstitutions != null && dtAssociatedInstitutions.Rows != null && dtAssociatedInstitutions.Rows.Count > 0 )
			{
				for ( int idx = 0; idx < dtAssociatedInstitutions.Rows.Count; idx++ )
				{
					DataRow drAssociatedInstitutions = dtAssociatedInstitutions.Rows[idx];
					//
					string institutionDivisionCode = Convert.ToString( drAssociatedInstitutions[TABLE.AssociatedInstitution.InstitutionDivisionCode] ).Trim();
					string divisionCode = Convert.ToString( drAssociatedInstitutions[TABLE.AssociatedInstitution.DivisionCode] ).Trim();
					//
					_associatedInstitutions.Add( institutionDivisionCode, divisionCode );
				}
			}
		}

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/19/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5818"> 
		///		<ExpectedInput>Valid HL7 message for Type and Screen.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA".</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5819"> 
		///		<ExpectedInput>Valid HL7 Message with an invalid Order Control Code of 'NN'.</ExpectedInput>
		///		<ExpectedOutput>HL7Exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="5959"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for ABO_RH for an existing patient 
		///		in VBECS with different demographic data to initiate a PatientChange entry.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA".</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6767"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for ABO_RH for a new patient.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6768"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for an Antibody Screen test
		///		for a new patient.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6769"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for a DAT.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6770"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for a TRW.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6771"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for an RBC component order.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6772"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for an FFP component order.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6773"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for CRYO component order.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="6774"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for Platelet component order.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="6775"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for Whole Blood component order.</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6776"> 
		///		<ExpectedInput>Valid HL7 Message with a new order for Other component Order</ExpectedInput>
		///		<ExpectedOutput>Non-null HL7OrgMessage with AckType of "AA"</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method is responsible for parsing the HL7 new order message passed into the constructor and inserting the new order in the VBECS database.
		/// </summary>
		/// <returns><see cref="HL7OrgMessage"/> containing a valid HL7 response message to the incoming order message.</returns>
		public HL7OrgMessage ProcessNewOrder()
		{
			if( _message == null )
				throw( new ArgumentNullException( "_message" ) );

			if( _message.OrderControlCode != "NW" )
				throw( new HL7Exception( "Order Control Code must be 'NW' to call CprsNewOrder.ProcessNewOrder" ) );

			LoadDataFromOmgMessage();
			
			// Determine type of order and get ID from appropriate reference table
			DataTable _orderUpdateData = ( OrderTable == ORDERED_TEST ) ? this.OrderedTest : this.OrderedComponent;
			// Reset the datatables before inserting/updating, if needed
			ResetOrderDataBeforeProcessingNewOrder();

			HL7AL.CprsOmgMessage.UpdateVistaPatientOrder(this.Patient, this.PatientTreatment, this.PatientOrder, _orderUpdateData);

			// Print the order alert, if needed, but don't send a reject acknowledgement if it fails.
			// Just write an event and send an email to the HL7 Administrator.
			try
			{
                //CR3574 - PrintPatientOrderAlert no longer static - to avoid printout corruption
				new BOL.PatientOrder().PrintPatientOrderAlert( this.PatientOrderGuid, this.CprsOrderNumber, "CPRS", this.DivisionCode );
			}
            // *** Fortify Justified Code ***
            // *** We catch all exceptions by design. We do not suppress all exceptions silently. We record into the log every time we catch exception. So we know every time it happens and can trouble shoot accordingly. All exceptions should be caught to prevent service from failure.  ***
            catch (Exception exc)
            {
                //EventLogAppender
                _eventsLogger.Error(HL7Utility.GetInnerExceptionFromException(exc));
                // CR 1928 - BNT 6/2/06 Removed HL7 Message text from email message.
                var messageBody = "Exception Message:\n" + HL7Utility.GetInnerExceptionFromException(exc);
                var mail = new HL7MailMessage();
                mail.SendMessage(this._intParms.InterfaceAdministratorEmail, PII                   ", "VBECS PrintPatientOrderAlert Error", messageBody, "smtp.DNS   ");
            }
            // *** Fortify Justified Code *** 

            return CprsOrgMessage.AppAcceptAck( _message, OrderControlCodes.OK, this.VbecsOrderNumber );
		}

		#endregion

		#region Private Classes

		/// <summary>
		/// Class DataTableDebugHelper
		/// </summary>
		private class DataTableDebugHelper
		{		
			/// <summary>
			/// Describe
			/// </summary>
			public static string Describe( string dtCodeName, DataTable dataTable )
			{
				if( dtCodeName == null )
					throw( new ArgumentNullException( "dtCodeName" ) );

				if( dataTable == null )
					throw( new ArgumentNullException( "dataTable" ) );

				DataTable dt = dataTable.Copy();			

				StringBuilder _sb = new StringBuilder();

				_sb.AppendFormat( "Describing DataTable {0}{1}", dtCodeName, Environment.NewLine );
				_sb.Append( DescribeTableWithXml( dt ) );
				_sb.Append( Environment.NewLine );			

				return _sb.ToString();
			}

			/// <summary>
			/// DescribeTableWithXml
			/// </summary>
			private static string DescribeTableWithXml( DataTable dt )
			{
				DataSet _ds = new DataSet( "DataTableDecriptor" );
				_ds.Tables.Add( dt );

				MemoryStream _ms = new MemoryStream( 10000 );
			
				XmlTextWriter _wtr = new XmlTextWriter( _ms, Encoding.ASCII );
				_wtr.Formatting = Formatting.Indented;			
				_ds.WriteXml( _wtr );
				_wtr.Flush();

				_ms.Seek( 0L, SeekOrigin.Begin );			
				StreamReader _rdr = new StreamReader( _ms );
                // *** Fortify Justified Code ***
                // *** This code uses MemoryStream set to 10000 bytes and so we cannot read past the end of the stream. ***
                string _result = _rdr.ReadToEnd();
                // *** Fortify Justified Code *** 


                _rdr.Close();
				_wtr.Close();
				_ms.Close();

				return _result;
			}
		}

		#endregion

		#region Private Methods


		/// <summary>
		/// ResetOrderDataBeforeProcessingNewOrder
		/// </summary>
		private void ResetOrderDataBeforeProcessingNewOrder()
		{
			if( !this._patientExists )
				return;

			this.Patient.Reset();

			if( !this._patientTreatmentExists )
				return;

			this.PatientTreatment.Reset();

			if( !this._patientOrderExists )
				return;

			this.PatientOrder.Reset();
		}

		/// <summary>
		/// GetVbecsOrderableId
		/// </summary>
		private int GetVbecsOrderableId(int cprsOrderableItemId)
		{
			if (cprsOrderableItemId < 1)
				throw( new HL7Exception("Orderable Item ID " + cprsOrderableItemId + "out of range") );
			switch( cprsOrderableItemId )
			{
				case ABORH_CPRS_ORDERABLE_ITEM_ID:
					return ABORH_ORDERABLE_TEST_ID;

				case ABS_CPRS_ORDERABLE_ITEM_ID:
					return ABS_ORDERABLE_TEST_ID;

				case DAT_CPRS_ORDERABLE_ITEM_ID:
					return DAT_ORDERABLE_TEST_ID;

				case TAS_CPRS_ORDERABLE_ITEM_ID:
					return TAS_ORDERABLE_TEST_ID;

				case TRW_CPRS_ORDERABLE_ITEM_ID:
					return TRW_ORDERABLE_TEST_ID;
		
				case RBC_CPRS_ORDERABLE_ITEM_ID:
					return RBC_COMPONENT_CLASS_ID;
		
				case FFP_CPRS_ORDERABLE_ITEM_ID:
					return FFP_COMPONENT_CLASS_ID;
		
				case CRYO_CPRS_ORDERABLE_ITEM_ID:
					return CRYO_COMPONENT_CLASS_ID;
		
				case PLT_CPRS_ORDERABLE_ITEM_ID:
					return PLT_COMPONENT_CLASS_ID;
		
				case OTHER_CPRS_ORDERABLE_ITEM_ID:
					return OTHER_COMPONENT_CLASS_ID;
		
				case WB_CPRS_ORDERABLE_ITEM_ID:
					return WB_COMPONENT_CLASS_ID;

				default:
					return -1;
			}
		}

		/// <summary>
		/// Gets all of the data for an Ordered Component request.
		/// </summary>
		private void GetOrderedComponentData()
		{

			string requiredUnitQuantity = HL7Utility.ConvertString( this.Message.TimingQuantity[0] );
			if (requiredUnitQuantity != "")
			{
				_drOrderedComponent[TABLE.OrderedComponent.RequiredUnitQuantity] = Convert.ToDecimal(requiredUnitQuantity);
			}

			string specimenActionCode = HL7Utility.ConvertString( this.Message.OBR[11] );

			if ( specimenActionCode != null )
			{
				switch (specimenActionCode)
				{
					case "L":
						_drOrderedComponent[TABLE.OrderedComponent.SpecimenStatusCode] = SPECIMEN_REQUIRED;
						break;

					case "O":
						_drOrderedComponent[TABLE.OrderedComponent.SpecimenStatusCode] = SPECIMEN_NOT_REQUIRED;
						break;

					case "A":
						_drOrderedComponent[TABLE.OrderedComponent.SpecimenStatusCode] = SPECIMEN_ALREADY_IN_BLOOD_BANK;
						break;

				}
			}
			//
			// Get the Specimen Source data including the Specimen UID and SpecimenCollectionText.
			string[] specimenSource = HL7Utility.ParseGetStringArray( this.Message.OBR[15], this.Message.Delimiters[1].ToString() );

			if ( specimenActionCode == SPECIMEN_ALREADY_IN_BLOOD_BANK )
			{
				_drOrderedComponent[TABLE.PatientSpecimen.SpecimenUid] = HL7Utility.ConvertString( specimenSource[0] );
			}
			else
			{
				_drOrderedComponent[TABLE.PatientSpecimen.SpecimenUid] = null;
			}

			_drOrderedComponent[TABLE.OrderedComponent.ComponentClassId] = GetVbecsOrderableId( Convert.ToInt16( this.Message.UniversalServiceID[0] ) );

			_drOrderedComponent[TABLE.OrderedComponent.OrderStatusCode] = PENDING_ORDER_STATUS_CODE;
			string sRequiredDateTime = HL7Utility.ConvertString( this.Message.TimingQuantity[4] );
			DateTime reqDateTime = DateTime.MinValue;
			if (sRequiredDateTime != null )
			{
				reqDateTime = HL7DateFormat.ConvertHL7DateTime(sRequiredDateTime);
				_drOrderedComponent[TABLE.OrderedComponent.RequiredDatetime] = reqDateTime;
			}

			_drOrderedComponent[TABLE.OrderedComponent.OrderSpecialInstructions] = HL7Utility.ConvertString( this.Message.UniversalServiceID[4] );

			_drOrderedComponent[TABLE.OrderedComponent.OrderedComponentComments] = GetOrderComments( this.Message.NTE );	
		}

		/// <summary>
		/// Get all of the data for an Ordered Test request.
		/// </summary>
		private void GetOrderedTestData()
		{
			_drOrderedTest[TABLE.OrderedTest.OrderUrgencyCode] = HL7Utility.ConvertString( this.Message.TimingQuantity[5] );
			_drOrderedTest[TABLE.OrderedTest.OrderStatusCode] = PENDING_ORDER_STATUS_CODE;

			_drOrderedTest[TABLE.OrderedTest.OrderedTestComments] = GetOrderComments( this.Message.NTE );

			DateTime reqResDate = DateTime.MinValue;
			string sRequiredResultsDate = HL7Utility.ConvertString( this.Message.TimingQuantity[4] );
			if ( sRequiredResultsDate != null )
			{
				reqResDate = HL7DateFormat.ConvertHL7DateTime(sRequiredResultsDate);
				_drOrderedTest[TABLE.OrderedTest.RequiredResultsDate] = reqResDate;
			}
			_drOrderedTest[TABLE.OrderedTest.OrderableTestId] = GetVbecsOrderableId( Convert.ToInt16( this.Message.UniversalServiceID[0] ) );
			
		}

		/// <summary>
		/// Get order comments from the NTE segment
		/// </summary>
		/// <param name="orderComments"></param>
		/// <returns></returns>
		private string GetOrderComments( IEnumerable orderComments )
		{
			StringBuilder sb = new StringBuilder();
			System.Collections.IEnumerator myEnumerator = orderComments.GetEnumerator();
			while ( myEnumerator.MoveNext() )
			{
                //CR 3554
			    if (myEnumerator != null && myEnumerator.Current != null && this.Message != null &&
			        this.Message.Delimiters != null && this.Message.Delimiters.Count() >= 1)
			    {
                    if (this.Message.Delimiters[0] == Char.MinValue)
                        throw new ApplicationException("Delimiters first element is null");
                    string de = new string(this.Message.Delimiters);
                    string delimit = de.Substring(0,1);
                    string sbNew = HL7Utility.ParseGetSegmentData(myEnumerator.Current.ToString(), delimit, 3);
                    sb.Append(sbNew);
			    }
			}
			return sb.ToString();
		}

		/// <summary>
		/// Check if patient exists by PatientIcn and VistaPatienID
		/// </summary>
		private void GetPatientGuidFromVbecs()
		{
			DataTable dtSearch = null;

			if( PatientIcn == "" )
			{
				dtSearch = HL7AL.CprsOmgMessage.GetHl7Patient( null, VistaPatientId );
			}
			else
			{
				dtSearch = HL7AL.CprsOmgMessage.GetHl7Patient( null, VistaPatientId );
			}

			if (dtSearch.Rows.Count > 0)	// Patient exists
			{
				this.PatientGuid = (Guid) dtSearch.Rows[0][TABLE.Patient.PatientGuid];
				// BNT 03/26/07: CR 2161 - Removed undocumented patient update functionality from new order parser.
				this._patientExists = true;
			}
			else
			{
				this.PatientGuid = System.Guid.NewGuid();
			}
		}

		/// <summary>
		/// GetPatientTreatmentGuidFromVbecs
		/// </summary>
		private void GetPatientTreatmentGuidFromVbecs()
		{
			DataTable dtSearch = HL7AL.CprsOmgMessage.GetHl7PatientTreatment(_dtPatientTreatment);

			if(dtSearch.Rows.Count > 0)
			{
				this.PatientTreatmentGuid = (Guid) dtSearch.Rows[0][TABLE.PatientTreatment.PatientTreatmentGuid];
				this._patientTreatmentExists = true;
			}
			else
			{
				this.PatientTreatmentGuid = System.Guid.NewGuid();
			}
		}

		/// <summary>
		/// GetPatientOrderGuidFromVbecs
		/// </summary>
		private void GetPatientOrderGuidFromVbecs()
		{
			DataTable dtSearch = HL7AL.CprsOmgMessage.GetPatientOrderbyOrderGroupNumber((string) _drPatientOrder[TABLE.PatientOrder.OrderGroupNumber]);

			if (dtSearch.Rows.Count > 0)	//Order exists
			{
				this.PatientOrderGuid = (Guid)dtSearch.Rows[0][TABLE.PatientOrder.PatientOrderGuid];
				this._patientOrderExists = true;
			}
			else
			{
				this.PatientOrderGuid = System.Guid.NewGuid();
			}
		}

		/// <summary>
		/// OrderedComponentExists
		/// </summary>
		private bool OrderedComponentExists()
		{
			DataTable dtSearch = HL7AL.CprsOmgMessage.GetOrderedComponentbyCPRSOrderNumber(CprsOrderNumber);

			if (dtSearch.Rows.Count > 0)
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// OrderedTestExists
		/// </summary>
		private bool OrderedTestExists()
		{
			DataTable dtSearch = HL7AL.CprsOmgMessage.GetOrderedTestbyCPRSOrderNumber(CprsOrderNumber);

			if (dtSearch.Rows.Count > 0)
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// InitializeDataTables
		/// </summary>
		private void InitializeDataTables()
		{
			_dtPatient = new DataTable();
			_drPatient = _dtPatient.NewRow();

			_dtPatientTreatment = new DataTable();
			_drPatientTreatment = _dtPatientTreatment.NewRow();

			_dtPatientOrder = new DataTable();
			_drPatientOrder = _dtPatientOrder.NewRow();

			_dtOrderedTest = new DataTable();
			_drOrderedTest = _dtOrderedTest.NewRow();

			_dtOrderedComponent = new DataTable();
			_drOrderedComponent = _dtOrderedComponent.NewRow();

			#region Patient DataTable

			_dtPatient.Columns.Add(TABLE.Patient.PatientGuid, typeof(Guid));
			_dtPatient.Columns.Add(TABLE.Patient.PatientFirstName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientLastName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientMiddleName);
			_dtPatient.Columns.Add(TABLE.Patient.PatientSsn);
			_dtPatient.Columns.Add(TABLE.Patient.SsnPseudoIndicator, typeof(byte));
			_dtPatient.Columns.Add(TABLE.Patient.PatientIcn);
			_dtPatient.Columns.Add(TABLE.Patient.VistaPatientId);
			_dtPatient.Columns.Add(TABLE.Patient.PatientDob, typeof(DateTime));
			_dtPatient.Columns.Add(TABLE.Patient.PatientDobCode);
			_dtPatient.Columns.Add(TABLE.Patient.PatientSexCode);
			_dtPatient.Columns.Add(TABLE.Patient.LastUpdateUser);
			_dtPatient.Columns.Add(TABLE.Patient.LastUpdateFunctionId, typeof(int));

			// Patient DataRow
			_drPatient[TABLE.Patient.PatientGuid] = System.Guid.Empty;
			_drPatient[TABLE.Patient.PatientFirstName] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientLastName] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientMiddleName] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientSsn] = System.DBNull.Value;
			_drPatient[TABLE.Patient.SsnPseudoIndicator] = 0;
			_drPatient[TABLE.Patient.PatientIcn] = System.DBNull.Value;
			_drPatient[TABLE.Patient.VistaPatientId] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientDob] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientDobCode] = System.DBNull.Value;
			_drPatient[TABLE.Patient.PatientSexCode] = System.DBNull.Value;
			_drPatient[TABLE.Patient.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
			_drPatient[TABLE.Patient.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;
			_dtPatient.Rows.Add(_drPatient);

			#endregion

			#region PatientTreatment DataTable

			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientTreatmentGuid, typeof(Guid));
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.TreatingSpecialtyCode);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientGuid, typeof(Guid));
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientTreatingFacility);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientLocation);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientRoomBed);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.InPatientIndicator, typeof(bool));
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.PatientAdmittingDiagnosis);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.LastUpdateUser);
			_dtPatientTreatment.Columns.Add(TABLE.PatientTreatment.LastUpdateFunctionId, typeof(int));

			// PatientTreatment DataRow
			_drPatientTreatment[TABLE.PatientTreatment.PatientTreatmentGuid] = System.Guid.Empty;
			_drPatientTreatment[TABLE.PatientTreatment.TreatingSpecialtyCode] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.PatientGuid] = System.Guid.Empty;
			_drPatientTreatment[TABLE.PatientTreatment.PatientTreatingFacility] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.PatientLocation] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.PatientRoomBed] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.InPatientIndicator] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.PatientAdmittingDiagnosis] = System.DBNull.Value;
			_drPatientTreatment[TABLE.PatientTreatment.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
			_drPatientTreatment[TABLE.PatientTreatment.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

			_dtPatientTreatment.Rows.Add(_drPatientTreatment);

			#endregion

			#region PatientOrder DataTable

			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.PatientOrderGuid, typeof(Guid));
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.PatientTreatmentGuid, typeof(Guid));
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderGroupNumber);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderPlacedDatetime, typeof(DateTime));
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderReceivedDatetime, typeof(DateTime));
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderId);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderLastName);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderFirstName);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingProviderMiddleInitial);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderingHospitalLocation);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredById);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByLastName);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByFirstName);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.OrderEnteredByMiddleInitial);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.RequestReason);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.InformedConsentIndicator);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.SurgeryName);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.LastUpdateUser);
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.LastUpdateFunctionId, typeof(int));
			_dtPatientOrder.Columns.Add(TABLE.PatientOrder.DivisionCode);
			
			// PatientOrder DataRow
			_drPatientOrder[TABLE.PatientOrder.PatientOrderGuid] = System.Guid.Empty;
			_drPatientOrder[TABLE.PatientOrder.PatientTreatmentGuid] = System.Guid.Empty;
			_drPatientOrder[TABLE.PatientOrder.OrderGroupNumber] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderPlacedDatetime] = System.DBNull.Value;
			// Moved OrderReceivedDatetime into Division set property to address
			// CR 1753 to get proper time zone related date/time.
			_drPatientOrder[TABLE.PatientOrder.OrderingProviderId] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderingProviderLastName] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderingProviderFirstName] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderingProviderMiddleInitial] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderingHospitalLocation] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderEnteredById] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderEnteredByLastName] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderEnteredByFirstName] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.OrderEnteredByMiddleInitial] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.RequestReason] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.InformedConsentIndicator] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.SurgeryName] = System.DBNull.Value;
			_drPatientOrder[TABLE.PatientOrder.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
			_drPatientOrder[TABLE.PatientOrder.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

			_dtPatientOrder.Rows.Add(_drPatientOrder);

			#endregion
			
			#region OrderedComponent DataTable

			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderedComponentGuid, typeof(Guid));
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.CprsOrderNumber);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.VbecsOrderNumber);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.RequiredUnitQuantity, typeof(decimal));
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.SpecimenStatusCode);
			_dtOrderedComponent.Columns.Add(TABLE.PatientSpecimen.SpecimenUid);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.ComponentClassId);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.PatientOrderGuid);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.PreOpIndicator);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderUrgencyCode);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderStatusCode);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.RequiredDatetime, typeof(DateTime));
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderSpecialInstructions);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.OrderedComponentComments);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.LabOrderNumber);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.LabTestId);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.DivisionCode);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.LastUpdateUser);
			_dtOrderedComponent.Columns.Add(TABLE.OrderedComponent.LastUpdateFunctionId, typeof(int));
			
			// OrderedComponent DataRow
			_drOrderedComponent[TABLE.OrderedComponent.OrderedComponentGuid] = System.Guid.NewGuid();
			_drOrderedComponent[TABLE.OrderedComponent.CprsOrderNumber] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.VbecsOrderNumber] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.RequiredUnitQuantity] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.SpecimenStatusCode] = System.DBNull.Value;
			_drOrderedComponent[TABLE.PatientSpecimen.SpecimenUid] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.ComponentClassId] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.PatientOrderGuid] = System.Guid.Empty;
			_drOrderedComponent[TABLE.OrderedComponent.PreOpIndicator] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.OrderUrgencyCode] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.OrderStatusCode] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.RequiredDatetime] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.OrderSpecialInstructions] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.OrderedComponentComments] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.LabOrderNumber] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.LabTestId] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.DivisionCode] = System.DBNull.Value;
			_drOrderedComponent[TABLE.OrderedComponent.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
			_drOrderedComponent[TABLE.OrderedComponent.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

			_dtOrderedComponent.Rows.Add(_drOrderedComponent);

			#endregion

			#region OrderedTest DataTable

			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.PatientOrderGuid, typeof(Guid));
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestGuid, typeof(Guid));
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.CprsOrderNumber);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.VbecsOrderNumber);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderUrgencyCode);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderStatusCode);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderedTestComments);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.RequiredResultsDate, typeof(DateTime));
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.OrderableTestId);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.LabOrderNumber);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.LabTestId);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.DivisionCode);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateUser);
			_dtOrderedTest.Columns.Add(TABLE.OrderedTest.LastUpdateFunctionId, typeof(int));
			
			// OrderedTest DataRow
			_drOrderedTest[TABLE.OrderedTest.PatientOrderGuid] = System.Guid.Empty;
			_drOrderedTest[TABLE.OrderedTest.OrderedTestGuid] = System.Guid.NewGuid();
			_drOrderedTest[TABLE.OrderedTest.CprsOrderNumber] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.VbecsOrderNumber] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.OrderUrgencyCode] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.OrderStatusCode] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.OrderedTestComments] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.RequiredResultsDate] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.OrderableTestId] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.LabOrderNumber] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.LabTestId] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.DivisionCode] = System.DBNull.Value;
			_drOrderedTest[TABLE.OrderedTest.LastUpdateUser] = LAST_UPDATE_USER_PARAMETER_VALUE;
			_drOrderedTest[TABLE.OrderedTest.LastUpdateFunctionId] = Common.UpdateFunction.HL7CprsInterface;

			_dtOrderedTest.Rows.Add(_drOrderedTest);

			#endregion

			// Load the Division this order is being placed for in order to
			// Determine if Division is supported by this instance of VBECS.
			// Address CR 1801.
			string[] division = HL7Utility.ParseGetStringArray( this.Message.ORC[17], this.Message.Delimiters[1].ToString() );
			string msgDivision = HL7Utility.ConvertString( division[0] ).Trim();
			//
			msgDivision = Convert.ToString( _associatedInstitutions[msgDivision] );
			if( msgDivision == string.Empty )
			{
                // EventLogAppender
                _eventsLogger.Error(EXCEPTION_ASSOCIATED_INSTITUTION_LOOKUP);
				//
				throw( new HL7Exception(EXCEPTION_ASSOCIATED_INSTITUTION_LOOKUP) );
			}
			//
            this.Division = new BOL.Division(msgDivision);
            if (!this.Division.IsActive)
            {
                throw (new HL7Exception("Division " + msgDivision + " is not active in this instance of VBECS."));
            }
            //CR3508 - removed try/catch around above block - all errors bubble up to CprsHL7Parser.ParseHL7Message() - which has a generic Exception trap

			this.OrderingHospitalLocation = HL7Utility.ConvertString( division[1] );
		}

		/// <summary>
		/// LoadDataFromOmgMessage
		/// </summary>
		private void LoadDataFromOmgMessage()
		{
			if( _message == null )
				throw( new ArgumentNullException( "_message" ) );
            //CR 3243
            //Check the value of processing id in the message
            if (this.Message.MSH[10] != String.Empty)
            {
                char[] procId = this.Message.MSH[10].ToCharArray();
                if (procId[0] != this._intParms.ProcessingId)
                {
                    throw (new HL7Exception("Conflicting Processing ID in order message, service is configured as " + this._intParms.ProcessingId + " and the order message contains a processing ID of " + procId[0]));
                }
            }


		    // What type of order is this?  Test or Component.
			this._orderTable = HL7AL.CprsOmgMessage.GetOrderTableTypeID( HL7Utility.ConvertString(this.Message.UniversalServiceID[0]) );
			// Make sure this is a valid order ID. 
			if( OrderTable == "ERROR" )
			{
				throw( new HL7Exception("Invalid Order ID in OBR:4:1") );
			}
			string[] cprsOrderNumber = HL7Utility.ParseGetStringArray( this.Message.OBR[2], this.Message.Delimiters[1].ToString() );
			this.CprsOrderNumber = HL7Utility.ConvertString( cprsOrderNumber[0] );

			// Check if order already exists and don't accept a New Order message for an existing order in VBECS.
			if( (OrderTable == ORDERED_TEST && OrderedTestExists()) || (OrderTable == ORDERED_COMPONENT && OrderedComponentExists()) )
			{
				throw( new HL7Exception("An Order already exists in VBECS for CPRS Order number " + CprsOrderNumber ) );
			}

			this.VbecsOrderNumber = VbecsHL7Bridge.GetNextVbecsOrderNumber();
			this.PatientSexCode = HL7Utility.ConvertString( this.Message.PID[8]);
			this.PatientDob = HL7DateFormat.ConvertHL7DateTime( this.Message.PID[7] );

			_drPatient[TABLE.Patient.PatientDobCode] = "V";
			if ( this.Message.PID[7].Substring(4,4) == "0000")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "B";
			}
			else if ( this.Message.PID[7].Substring(4,2) == "00")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "M";
			}
			else if ( this.Message.PID[7].Substring(6,2) == "00")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "D";
			}

			string[] patientIdList = HL7Utility.ParseGetStringArray( this.Message.PID[3], this.Message.Delimiters[2].ToString() );
			for (int i = 0; i < patientIdList.Length ; i++ )
			{
				string[] idComponents = HL7Utility.ParseGetStringArray( patientIdList[i], this.Message.Delimiters[1].ToString() );
				switch(idComponents[4])
				{
					case "NI":
						_drPatient[TABLE.Patient.PatientIcn] = HL7Utility.ConvertString( idComponents[0] );
						break;

					case "SS":
						string patientSsn = HL7Utility.ConvertString( idComponents[0] );
						if( patientSsn.EndsWith("P"))
						{
							_drPatient[TABLE.Patient.PatientSsn] = patientSsn.Remove(patientSsn.Length-1, 1);
							_drPatient[TABLE.Patient.SsnPseudoIndicator] = 1;
						}
						else
						{
							_drPatient[TABLE.Patient.PatientSsn] = patientSsn;
						}
						break;

					case "PI":
						if (idComponents[0] == null)
						{
							throw new HL7Exception("VistA Patient ID not provided");
						}
						else
						{
							_drPatient[TABLE.Patient.VistaPatientId] = HL7Utility.ConvertString( idComponents[0] );
						}
						break;
				}
			}

			string[] patientName = HL7Utility.ParseGetStringArray( this.Message.PID[5], this.Message.Delimiters[1].ToString() );
			for (int i = 0; i < patientName.Length; i++ )
			{
				if (i == 0)
				{
					this.PatientLastName = HL7Utility.ConvertString( patientName[0] );
				}
				if (i == 1)
				{
					this.PatientFirstName = HL7Utility.ConvertString( patientName[1] );
				}
				if (i == 2)
				{
					this.PatientMiddleName = HL7Utility.ConvertString( patientName[2] );
				}
			}

			_drPatient[TABLE.Patient.PatientDobCode] = "V";
			if ( this.Message.PID[7].Substring(4,4) == "0000")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "B";
			}
			else if ( this.Message.PID[7].Substring(4,2) == "00")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "M";
			}
			else if ( this.Message.PID[7].Substring(6,2) == "00")
			{
				_drPatient[TABLE.Patient.PatientDobCode] = "D";
			}

			if( this.Message.ORC.Length > 19 )
			{
				string[] informedConsent = HL7Utility.ParseGetStringArray( this.Message.ORC[20], this.Message.Delimiters[1].ToString() );
				if( informedConsent[0] != "U" )
				{
					_drPatientOrder[TABLE.PatientOrder.InformedConsentIndicator] = informedConsent[0];
				}
			}

			_drPatientTreatment[TABLE.PatientTreatment.TreatingSpecialtyCode] = HL7Utility.ConvertString( this.Message.PV1[10] );

			string[] patientLocation = HL7Utility.ParseGetStringArray( this.Message.PV1[3], this.Message.Delimiters[1].ToString() );
			for (int i =0; i < patientLocation.Length; i++)
			{
				if (i == 0)
				{
					_drPatientTreatment[TABLE.PatientTreatment.PatientLocation] = HL7Utility.ConvertString( patientLocation[0] );
				}
				if (i == 1)
				{
					_drPatientTreatment[TABLE.PatientTreatment.PatientRoomBed] = HL7Utility.ConvertString( patientLocation[1] );
				}
			}
			string[] patientDiagnosis = HL7Utility.ParseGetStringArray( this.Message.DG1[3], this.Message.Delimiters[1].ToString() );
			_drPatientTreatment[TABLE.PatientTreatment.PatientAdmittingDiagnosis] = HL7Utility.ConvertString( patientDiagnosis[4] );

			// The InPatientIndicator comes over as either I = Inpatient, or O = Outpatient.  Need to change to bit.
			string inPatientIndicator = HL7Utility.ConvertString( this.Message.PV1[2] );
			switch(inPatientIndicator)
			{
				case "I":
					_drPatientTreatment[TABLE.PatientTreatment.InPatientIndicator] = 1;
					break;
				case "O":
					_drPatientTreatment[TABLE.PatientTreatment.InPatientIndicator] = 0;
					break;
			}
			
			string[] orderGroupNumber = HL7Utility.ParseGetStringArray( this.Message.ORC[4], this.Message.Delimiters[1].ToString() );
			_drPatientOrder[TABLE.PatientOrder.OrderGroupNumber] = HL7Utility.ConvertString( orderGroupNumber[0] );

			string sOrderPlacedDatetime = HL7Utility.ConvertString( this.Message.ORC[9] );
			if (sOrderPlacedDatetime != "")
			{
				DateTime orderPlacedDate = HL7DateFormat.ConvertHL7DateTime(sOrderPlacedDatetime);
				_drPatientOrder[TABLE.PatientOrder.OrderPlacedDatetime] = orderPlacedDate;
			}
			string[] orderingProvider = HL7Utility.ParseGetStringArray( this.Message.ORC[12], this.Message.Delimiters[1].ToString() );
			for (int i = 0; i < orderingProvider.Length; i++ )
			{
				if (i == 0)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderingProviderId] = HL7Utility.ConvertString( orderingProvider[0] );
				}
				if (i == 1)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderingProviderLastName] = HL7Utility.ConvertString( orderingProvider[1] );
				}
				if (i == 2)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderingProviderFirstName] = HL7Utility.ConvertString( orderingProvider[2] );
				}
				if (i == 3)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderingProviderMiddleInitial] = HL7Utility.ConvertString( orderingProvider[3] );
				}
			}

			string[] enteredBy = HL7Utility.ParseGetStringArray( this.Message.ORC[10], this.Message.Delimiters[1].ToString() );
			for (int i = 0; i < enteredBy.Length; i++ )
			{
				if (i == 0)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderEnteredById] = HL7Utility.ConvertString( enteredBy[0] );
				}
				if (i == 1)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderEnteredByLastName] = HL7Utility.ConvertString( enteredBy[1] );
				}
				if (i == 2)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderEnteredByFirstName] = HL7Utility.ConvertString( enteredBy[2] );
				}
				if (i == 3)
				{
					_drPatientOrder[TABLE.PatientOrder.OrderEnteredByMiddleInitial] = HL7Utility.ConvertString( enteredBy[3] );
				}
			}

			_drPatientOrder[TABLE.PatientOrder.RequestReason] = HL7Utility.ParseGetSegmentData(this.Message.ORC[16],this.Message.Delimiters[1].ToString(),1);

			string[] labOrderNumber = HL7Utility.ParseGetStringArray(HL7Utility.ConvertString( this.Message.OBR[13]), ";" );
			if ( labOrderNumber.Length > 0 )
			{
				if( labOrderNumber[0] == string.Empty )
				{
					throw( new HL7Exception("Missing Lab Order ID in OBR:13:1") );

				}
				this.LabOrderNumber = labOrderNumber[0];
				if( labOrderNumber.Length == 4 )
				{
					this.LabTestId = labOrderNumber[3];
				}
			}
			else
			{
				throw( new HL7Exception("Missing Lab Order ID in OBR:13:1") );
			}

			string orderUrgencyCode = HL7Utility.ConvertString( this.Message.TimingQuantity[5] );
			if (orderUrgencyCode == PRE_OP_ORDER)
			{
				_drOrderedComponent[TABLE.OrderedComponent.PreOpIndicator] = "1";
				_drPatientOrder[TABLE.PatientOrder.SurgeryName] = HL7Utility.ConvertString( this.Message.TimingQuantity[7] );
			}
			else { _drOrderedComponent[TABLE.OrderedComponent.PreOpIndicator] = "0";}

			_drOrderedComponent[TABLE.OrderedComponent.OrderUrgencyCode] = orderUrgencyCode;

			GetPatientGuidFromVbecs();
			GetPatientTreatmentGuidFromVbecs();
			GetPatientOrderGuidFromVbecs();

			if( OrderTable == ORDERED_TEST )
			{
				GetOrderedTestData();
			}
			else
			{
				GetOrderedComponentData();
			}
		}

		#endregion

		#region Properties

		/// <summary>
		/// Get/Set Message
		/// </summary>
		private HL7OmgMessage Message
		{
			get
			{
				return _message;
			}
		}

		/// <summary>
		/// Get/Set PatientFirstName
		///Deleted Getter for defect# 286500 on 3/21/2016 By Russell Stephenson
		/// </summary>
		private string PatientFirstName
		{
			set
			{
				_drPatient[TABLE.Patient.PatientFirstName] = value;
			}
		}

		/// <summary>
		/// Get/Set PatientLastName
		/// </summary>
		private string PatientLastName
		{
			set
			{
				_drPatient[TABLE.Patient.PatientLastName] = value;
			}
		}

		/// <summary>
		/// Get/Set PatientMiddleName
		/// </summary>
		private string PatientMiddleName
		{
			set
			{
				_drPatient[TABLE.Patient.PatientMiddleName] = value;
			}
		}

		/// <summary>
		/// Get/Set PatientIcn
		/// </summary>
		private string PatientIcn
		{
			get
			{
				return StDbNullConvert.ToString(_drPatient[TABLE.Patient.PatientIcn]);
			}
		}


		/// <summary>
		/// Get/Set PatientDob
		/// </summary>
		private DateTime PatientDob
		{
			set
			{
				_drPatient[TABLE.Patient.PatientDob] = value;
			}
		}

		/// <summary>
		/// Get/Set PatientSexCode
		/// </summary>
		private string PatientSexCode
		{
			set
			{
				_drPatient[TABLE.Patient.PatientSexCode] = value;
			}
		}

		/// <summary>
		/// Get/Set VistaPatientId
		/// </summary>
		private string VistaPatientId
		{
			get
			{
				return StDbNullConvert.ToString(_drPatient[TABLE.Patient.VistaPatientId]);
			}
		}

		/// <summary>
		/// Get/Set LabOrderNumber
		/// </summary>
		private string LabOrderNumber
		{
			set
			{
				_drOrderedTest[TABLE.OrderedTest.LabOrderNumber] = value;
				_drOrderedComponent[TABLE.OrderedComponent.LabOrderNumber] = value;
			}
		}

		/// <summary>
		/// Get/Set LabTestId
		/// </summary>
		private string LabTestId
		{
			set
			{
				_drOrderedTest[TABLE.OrderedTest.LabTestId] = value;
				_drOrderedComponent[TABLE.OrderedComponent.LabTestId] = value;
			}
		}

		/// <summary>
		/// Get/Set CprsOrderNumber
		/// </summary>
		private string CprsOrderNumber
		{
			get
			{
				if (this.OrderTable == ORDERED_TEST && !_drOrderedTest.IsNull(TABLE.OrderedTest.CprsOrderNumber))
				{
					return StDbNullConvert.ToString(_drOrderedTest[TABLE.OrderedTest.CprsOrderNumber]);
				}
				else if (this.OrderTable == ORDERED_COMPONENT && !_drOrderedTest.IsNull(TABLE.OrderedComponent.CprsOrderNumber))
				{
					return StDbNullConvert.ToString(_drOrderedComponent[TABLE.OrderedComponent.CprsOrderNumber]);
				}
				else
				{
					throw( new HL7Exception( "CPRS Order Number not found" ) );
				}
			}
			set
			{
				_drOrderedTest[TABLE.OrderedTest.CprsOrderNumber] = value;
				_drOrderedComponent[TABLE.OrderedComponent.CprsOrderNumber] = value;
			}
		}

		/// <summary>
		/// Get/Set VbecsOrderNumber
		/// </summary>
		private int VbecsOrderNumber
		{
			get
			{
				if (this.OrderTable == ORDERED_TEST && !_drOrderedTest.IsNull(TABLE.OrderedTest.VbecsOrderNumber))
				{
					return Convert.ToInt32(_drOrderedTest[TABLE.OrderedTest.VbecsOrderNumber].ToString());
				}
				else if (this.OrderTable == ORDERED_COMPONENT && !_drOrderedTest.IsNull(TABLE.OrderedComponent.VbecsOrderNumber))
				{
					return Convert.ToInt32(_drOrderedComponent[TABLE.OrderedComponent.VbecsOrderNumber].ToString());
				}
				else
				{
					throw( new HL7Exception( "Error generating VBECS Order Number" ) );
				}
			}
			set
			{
				_drOrderedTest[TABLE.OrderedTest.VbecsOrderNumber] = value;
				_drOrderedComponent[TABLE.OrderedComponent.VbecsOrderNumber] = value;
			}
		}

		/// <summary>
		/// Get/Set DivisionCode
		/// </summary>
		private string DivisionCode
		{
			get
			{
				if (this.OrderTable == ORDERED_TEST && !_drOrderedTest.IsNull(TABLE.OrderedTest.DivisionCode))
				{
					return _drOrderedTest[TABLE.OrderedTest.DivisionCode].ToString();
				}
				else if (this.OrderTable == ORDERED_COMPONENT && !_drOrderedTest.IsNull(TABLE.OrderedComponent.DivisionCode))
				{
					return _drOrderedComponent[TABLE.OrderedComponent.DivisionCode].ToString();
				}
				else
				{
					throw( new HL7Exception( "Error identifying DivisionCode" ) );
				}
			}
			set
			{
				_drPatientOrder[TABLE.PatientOrder.DivisionCode] = value;
				_drOrderedTest[TABLE.OrderedTest.DivisionCode] = value;
				_drOrderedComponent[TABLE.OrderedComponent.DivisionCode] = value;
			}
		}

		/// <summary>
		/// Get/Set Division
		/// </summary>
		private BOL.Division Division
		{
			get
			{
				return _division;
			}
			set
			{
				_division = value;
				// Addressed 1753 to get the current division time.
				_drPatientOrder[TABLE.PatientOrder.OrderReceivedDatetime] = BOL.VBECSDateTime.GetDivisionCurrentDateTime(value.DivisionCode);
				this.DivisionCode = value.DivisionCode;
			}
		}

		/// <summary>
		/// Get/Set OrderingHospitalLocation
		/// </summary>
		private string OrderingHospitalLocation
		{
			set
			{
				_drPatientTreatment[TABLE.PatientTreatment.PatientTreatingFacility] = value;
				_drPatientOrder[TABLE.PatientOrder.OrderingHospitalLocation] = value;
			}
		}

		/// <summary>
		/// Get/Set OrderTable
		/// </summary>
		private string OrderTable
		{
			get
			{
				return this._orderTable;
			}
		}

		/// <summary>
		/// Get/Set PatientGuid
		/// </summary>
		private Guid PatientGuid
		{
			set
			{
				_drPatient[TABLE.Patient.PatientGuid] = value;
				_drPatientTreatment[TABLE.PatientTreatment.PatientGuid] = value;
				_patientGuid = value;
			}
		}

		/// <summary>
		/// Get/Set PatientTreatmentGuid
		/// </summary>
		private Guid PatientTreatmentGuid
		{
			set
			{
				_drPatientTreatment[TABLE.PatientTreatment.PatientTreatmentGuid] = value;
				_drPatientOrder[TABLE.PatientOrder.PatientTreatmentGuid] = value;
				_patientTreatmentGuid = value;
			}
		}

		/// <summary>
		/// Get/Set PatientOrderGuid
		/// </summary>
		private Guid PatientOrderGuid
		{
			get
			{
				return _patientOrderGuid;
			}
			set
			{
				_drPatientOrder[TABLE.PatientOrder.PatientOrderGuid] = value;
				_drOrderedTest[TABLE.OrderedTest.PatientOrderGuid] = value;
				_drOrderedComponent[TABLE.OrderedComponent.PatientOrderGuid] = value;
				_patientOrderGuid = value;
			}
		}

		/// <summary>
		/// Get/Set Patient
		/// </summary>
		private DataTable Patient
		{
			get
			{
				return this._dtPatient;
			}
		}

		/// <summary>
		/// Get/Set PatientTreatment
		/// </summary>
		private DataTable PatientTreatment
		{
			get
			{
				return this._dtPatientTreatment;
			}
		}

		/// <summary>
		/// Get/Set PatientOrder
		/// </summary>
		private DataTable PatientOrder
		{
			get
			{
				return this._dtPatientOrder;
			}
		}

		/// <summary>
		/// Get/Set OrderedTest
		/// </summary>
		private DataTable OrderedTest
		{
			get
			{
				return this._dtOrderedTest;
			}
		}

		/// <summary>
		/// Get/Set OrderedComponent
		/// </summary>
		private DataTable OrderedComponent
		{
			get
			{
				return this._dtOrderedComponent;
			}
		}


		#endregion
	}
}
